<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\HasMany;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    protected $table = 'users';
    protected $primaryKey = 'User_id';
    public $timestamps = true;

    protected $fillable = [
        'Email',
        'phone', // Added as new field
        'province',
        'latitude',
        'longitude',
        'has_used_trial',
        'Password_hash',
        'User_type',
        'Status',
        'location', // New field
        'is_active', // New field
        'last_login', // New field
    ];

    protected $casts = [
        'Created_at' => 'datetime',
        'Updated_at' => 'datetime',
        'User_type' => 'string',
        'Status' => 'string',
        'latitude' => 'float',
        'longitude' => 'float',
        'has_used_trial' => 'boolean',
        'is_active' => 'boolean', // New cast
    ];

    protected $hidden = [
        'Password_hash',
        'remember_token',
    ];

    public function getAuthPassword()
    {
        return $this->Password_hash;
    }

    public function roles(): BelongsToMany
    {
        return $this->belongsToMany(Role::class, 'user_roles', 'User_id', 'Role_id');
    }

    public function candidateProfile(): HasOne
    {
        return $this->hasOne(CandidateProfile::class, 'User_id', 'User_id');
    }

    public function companyProfile(): HasOne
    {
        return $this->hasOne(CompanyProfile::class, 'User_id', 'User_id');
    }

    public function skills(): HasMany
    {
        return $this->hasMany(UserSkill::class, 'User_id', 'User_id');
    }

    public function certifications(): HasMany
    {
        return $this->hasMany(Certification::class, 'User_id', 'User_id');
    }

    public function currentSubscription(): HasOne
    {
        return $this->hasOne(Subscription::class, 'User_id', 'User_id')
            ->where('Status', 'active')
            ->where(function ($query) {
                $query->where('End_date', '>=', now())->orWhereNull('End_date');
            });
    }

    /**
     * Check if the user has a specific role.
     */
    public function hasRole($role)
    {
        return $this->roles()->where('Name', $role)->exists();
    }

    /**
     * Check if the user has a specific permission.
     */
    public function hasPermission($permission)
    {
        return $this->roles->pluck('Permissions')->flatten()->contains($permission);
    }

    /**
     * Check if the user has any of the specified roles.
     */
    public function hasAnyRole(...$roles)
    {
        $roleNames = collect($roles)->flatten()->all();
        return $this->roles()->whereIn('Name', $roleNames)->exists();
    }

    /**
     * Update last login timestamp.
     */
    public function updateLastLogin()
    {
        $this->update(['last_login' => now()]);
    }
}
